/*
 *  $Id: build.c,v 1.5 1997/05/19 12:29:58 mj Exp $
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *  Copyright (C) 1997 Martin Mares
 */

/*
 * This file builds a disk-image from three different files:
 *
 * - bootsect: exactly 512 bytes of 8086 machine code, loads the rest
 * - setup: 8086 machine code, sets up system parm
 * - system: 80386 code for actual system
 *
 * It does some checking that all files are of the correct type, and
 * just writes the result to stdout, removing headers and padding to
 * the right amount. It also writes some system data to stderr.
 */

/*
 * Changes by tytso to allow root device specification
 * High loaded stuff by Hans Lermen & Werner Almesberger, Feb. 1996
 * Cross compiling fixes by Gertjan van Wingerde, July 1996
 * Rewritten by Martin Mares, April 1997
 * Modified for binary file by Yann Droneaud <lch@multimania.com>, July 1999
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <unistd.h>
#include <fcntl.h>
#include <asm/boot.h>

typedef unsigned char byte;
typedef unsigned short word;
typedef unsigned long u32;

#define DEFAULT_MAJOR_ROOT 0
#define DEFAULT_MINOR_ROOT 0

/* Minimal number of setup sectors (see also bootsect.S) */
#define SETUP_SECTS 4

byte buf[1024];
int fd;
int is_big_kernel;
int is_minix;
char *program_name;


void die(const char * str, ...)
{
  va_list args;
  va_start(args, str);
  vfprintf(stderr, str, args);
  fputc('\n', stderr);
  exit(1);
}

/* Reading a binary file (produce with objcopy -O binary) */
void binary_open(const char *name)
{
  if ((fd = open(name, O_RDONLY, 0)) < 0)
    die("Unable to open `%s': %m", name);
}


/* Reading of ld86 output (Minix format) */
#define MINIX_HEADER_LEN 32
void minix_open(const char *name)
{
  static byte hdr[] = { 0x01, 0x03, 0x10, 0x04, 0x20, 0x00, 0x00, 0x00 };
  static u32 *lb = (u32 *) buf;
  
  if ((fd = open(name, O_RDONLY, 0)) < 0)
    die("Unable to open `%s': %m", name);
  if (read(fd, buf, MINIX_HEADER_LEN) != MINIX_HEADER_LEN)
    die("%s: Unable to read header", name);
  if (memcmp(buf, hdr, sizeof(hdr)) || lb[5])
    die("%s: Non-Minix header", name);
  if (lb[3])
    die("%s: Illegal data segment",name);
  if (lb[4])
    die("%s: Illegal bss segment",name);
  if (lb[7])
    die("%s: Illegal symbol table",name);
}

void usage(void)
{
  die(\
"Usage: %s [options] bootsector setup system [rootdevice] [> image]\n"\
"Build a bootable kernel image \n"\
"Options in:\n"\
" -b\t--big-kernel\tBuild a big image (bzImage)\n"\
" -n\t--normal-kernel\tBuild a normal image (zImage)\n"\
" -M\t--minix\t\tRead object files in Minix format\n"\
" -B\t--binary\tRead object files in Binary format\n"\
" -h,-?\t--help\t\tShow this help\n"\
" \t--version\tShow version\n"\
,program_name);
}

void version(void)
{
  die("%s version of July 1999",program_name);
}

int main(int argc, char ** argv)
{
  char *bootsector_filename = NULL;
  char *setup_filename = NULL;
  char *system_filename = NULL;
  char *rootdevice_name = NULL;
  char *argument;

  int minusminus = 0;
  int argument_count = 0;

  unsigned int i, c, sz, setup_sectors;
  u32 sys_size;
  byte major_root, minor_root;
  struct stat sb;


  is_big_kernel = 0;
  is_minix = 1;

  if (strrchr (argv[0], '/') != NULL)
    program_name = strrchr (argv[0], '/') + 1;
  else
    program_name = argv[0];
  
  minusminus = 0;
  for(i = 1; i < argc; i++)
    {
      argument = argv[i];
      if ((minusminus != 1) && (*argument == '-'))
	{
	  if (!strcmp(argument,"--"))
	    minusminus = 1;
	  else
	    if (!strcmp(argument,"-b") || !strcmp(argument,"--big-kernel"))
	      is_big_kernel = 1;
	    else
	      if (!strcmp(argument,"-n") || !strcmp(argument,"--normal-kernel"))
		is_big_kernel = 0;
	      else
		if (!strcmp(argument,"-M") || !strcmp(argument, "--minix"))
		  is_minix=1;
		else
		  if (!strcmp(argument,"-B") || !strcmp(argument, "--binary"))
		    is_minix=0;
		  else
		    if (!strcmp(argument,"-h") || !strcmp(argument,"-?") || !strcmp(argument, "--help"))
		      usage();
		    else
		      if (!strcmp(argument,"--version"))
			version();
		      else
			die("Unknown option '%s'\nTry %s --help for more information\n",argument,program_name);
	}
      else
	{
	  switch (argument_count)
	    {
	    case 0: 
	      bootsector_filename = argument;
	      break;
	    case 1:
	      setup_filename = argument;
	      break;
	    case 2:
	      system_filename = argument;
	      break;
	    case 3:
	      rootdevice_name = argument;
	      break;
	    default:
	      usage();
	      break;
	    }
	  
	  argument_count ++;
	}
    }


  if (bootsector_filename == NULL || setup_filename == NULL || system_filename == NULL)
    die("No input files\nTry %s --help for information",program_name);

  if (rootdevice_name != NULL)
    {  
      if (!strcmp(rootdevice_name, "CURRENT")) 
	{
	  if (stat("/", &sb)) 
	    {
	      die("Couldn't stat /: %m ");
	    }
	  major_root = major(sb.st_dev);
	  minor_root = minor(sb.st_dev);
	} 
      else 
	if (!strcmp(rootdevice_name, "FLOPPY")) 
	  {
	    major_root = 0;
	    minor_root = 0;
	  }
	else 
	  {
	    if (stat(rootdevice_name, &sb)) 
	      {
		die("Couldn't stat root device %s: %m",rootdevice_name);
	      }
	    major_root = major(sb.st_rdev);
	    minor_root = minor(sb.st_rdev);
	  } 
    } 
  else 
    {
      major_root = DEFAULT_MAJOR_ROOT;
      minor_root = DEFAULT_MINOR_ROOT;
    }

  fprintf(stderr, "Root device is (%d, %d)\n", major_root, minor_root);
  
  /* Copy the boot sector */
  if (is_minix)
    minix_open(bootsector_filename);
  else
    binary_open(bootsector_filename);

  i = read(fd, buf, sizeof(buf));
  fprintf(stderr,"Boot sector is %d bytes\n",i);
  if (i != 512)
    die("Boot block must be exactly 512 bytes");
  if (buf[510] != 0x55 || buf[511] != 0xaa)
    die("Boot block hasn't got boot flag (0xAA55)");
  buf[508] = minor_root;
  buf[509] = major_root;
  if (write(1, buf, 512) != 512)
    die("Write call failed");
  close (fd);

  /* Copy the setup code */
  if (is_minix)
    minix_open(setup_filename);
  else
    binary_open(setup_filename);

  for (i=0 ; (c=read(fd, buf, sizeof(buf)))>0 ; i+=c )
    if (write(1, buf, c) != c)
      die("Write call failed");
  if (c != 0)
    die("read-error in %s: %m",setup_filename);
  close (fd);
  
  setup_sectors = (i + 511) / 512;		    /* Pad unused space with zeros */
  /* for compatibility with ancient versions of LILO */
  if (setup_sectors < SETUP_SECTS)
    setup_sectors = SETUP_SECTS;
  fprintf(stderr, "Setup is %d bytes\n", i);
  memset(buf, 0, sizeof(buf));
  while (i < setup_sectors * 512) 
    {
      c = setup_sectors * 512 - i;
      if (c > sizeof(buf))
	c = sizeof(buf);
      if (write(1, buf, c) != c)
	die("Write call failed");
      i += c;
    }
  
  if ((fd = open(system_filename, O_RDONLY, 0)) < 0)	    /* Copy the image itself */
    die("Unable to open `%s': %m", system_filename);
  if (fstat (fd, &sb))
    die("Unable to stat `%s': %m", system_filename);
  sz = sb.st_size;
  fprintf (stderr, "System is %d kB\n", sz/1024);
  sys_size = (sz + 15) / 16;
  if (sys_size > (is_big_kernel ? 0xffff : DEF_SYSSIZE))
    die("System is too big. Try using %smodules.",is_big_kernel ? "" : "bzImage or ");
  while (sz > 0) 
    {
      int l, n;
      
      l = (sz > sizeof(buf)) ? sizeof(buf) : sz;
      if ((n=read(fd, buf, l)) != l)
	{
	  if (n < 0)
	    die("Error reading %s: %m", system_filename);
	  else
	    die("%s: Unexpected EOF", system_filename);
	}
      if (write(1, buf, l) != l)
	die("Write failed");
      sz -= l;
    }
  close(fd);
  
  if (lseek(1, 497, SEEK_SET) != 497)		    /* Write sizes to the boot sector */
    die("Output: seek failed");
  buf[0] = setup_sectors;
  if (write(1, buf, 1) != 1)
    die("Write of setup sector count failed");
  if (lseek(1, 500, SEEK_SET) != 500)
    die("Output: seek failed");
  buf[0] = (sys_size & 0xff);
  buf[1] = ((sys_size >> 8) & 0xff);
  if (write(1, buf, 2) != 2)
    die("Write of image length failed");
  
  return 0;					    /* Everything is OK */
}
